"""
Feature settings module.


Copyright (c) 2023 Proton AG

This file is part of Proton VPN.

Proton VPN is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Proton VPN is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with ProtonVPN.  If not, see <https://www.gnu.org/licenses/>.
"""
from typing import TYPE_CHECKING

from gi.repository import Gtk, GObject
from proton.vpn.app.gtk.widgets.main.confirmation_dialog import ConfirmationDialog
from proton.vpn.core.settings import NetShield
from proton.vpn.app.gtk.controller import Controller
from proton.vpn.app.gtk.widgets.headerbar.menu.settings.common import (
    BaseCategoryContainer, ComboboxWidget, ToggleWidget,
    ReactiveSettingContainer
)
from proton.vpn.app.gtk.widgets.headerbar.menu.settings.custom_dns import CustomDNSWidget
from proton.vpn.app.gtk.widgets.headerbar.menu.settings.kill_switch import KillSwitchWidget
from proton.vpn.app.gtk.widgets.headerbar.menu.settings.split_tunneling import SplitTunnelingToggle

if TYPE_CHECKING:
    from proton.vpn.app.gtk.widgets.headerbar.menu.settings.settings_window import \
        SettingsWindow


class FeatureSettings(BaseCategoryContainer, ReactiveSettingContainer):  # noqa: E501 # pylint: disable=line-too-long, too-many-instance-attributes
    """Settings related to connection are all grouped under this class."""
    CATEGORY_NAME = "Features"
    NETSHIELD_LABEL = "NetShield"
    NETSHIELD_DESCRIPTION = "Protect yourself from ads, malware, and trackers "\
        "on websites and apps."
    PORT_FORWARDING_LABEL = "Port forwarding"
    PORT_FORWARDING_DESCRIPTION = "Bypass firewalls to connect to P2P servers "\
        "and devices on your local network."
    PORT_FORWARDING_DESCRIPTION_LEARN_MORE = "Bypass firewalls to connect to P2P servers "\
        "and devices on your local network. "\
        "<a href=\"https://protonvpn.com/support/port-forwarding/#linux\">Learn more</a>"
    PORT_FORWARDING_SETUP_GUIDE = "Follow our "\
        "<a href=\"https://protonvpn.com/support/port-forwarding-manual-setup/"\
        "#how-to-use-port-forwarding\">guide</a>"\
        " to set it up."
    SWITCH_KILLSWITCH_IF_CONNECTION_ACTIVE_DESCRIPTION = "Kill switch selection "\
        "is disabled while VPN is active. Disconnect to make changes."

    def __init__(self, controller: Controller, settings_window: "SettingsWindow"):
        super().__init__(self.CATEGORY_NAME)
        self._controller = controller
        self._settings_window = settings_window
        self.netshield = None
        self.killswitch = None
        self.port_forwarding = None
        self.split_tunneling = None

    def build_ui(self):
        """Builds the UI, invoking all necessary methods that are
        under this category."""
        self.build_netshield()
        self.build_killswitch()
        self.build_port_forwarding()
        if (
            self._controller.split_tunneling_available
            and self._controller.feature_flags.get("DisplaySplitTunneling")
        ):
            self.build_split_tunneling()

    def build_netshield(self):
        """Builds and adds the `netshield` setting to the widget.
        It takes into consideration the `clientconfig` value and if
        the user has the expected `tier` to be used. If the user has a
        lower tier then required then an upgrade UI is displayed.
        """
        def on_combobox_changed(combobox: Gtk.ComboBoxText, combobox_widget: ComboboxWidget):
            model = combobox.get_model()
            treeiter = combobox.get_active_iter()
            netshield = int(model[treeiter][1])
            combobox_widget.save_setting(netshield)
            self.emit("netshield-setting-changed", netshield)

        netshield_options = [
            (str(NetShield.NO_BLOCK.value), "Off"),
            (str(NetShield.BLOCK_MALICIOUS_URL.value), "Block Malware"),
            (str(NetShield.BLOCK_ADS_AND_TRACKING.value), "Block ads, trackers and malware"),
        ]
        self.netshield = ComboboxWidget(
            controller=self._controller,
            title=self.NETSHIELD_LABEL,
            description=self.NETSHIELD_DESCRIPTION,
            setting_name="settings.features.netshield",
            combobox_options=netshield_options,
            requires_subscription_to_be_active=True,
            callback=on_combobox_changed
        )
        self.append(self.netshield)

    def build_killswitch(self):
        """Builds and adds the `killswitch` setting to the widget."""
        self.killswitch = KillSwitchWidget.build(self._controller)
        self.append(self.killswitch)

    def build_port_forwarding(self):
        """Builds and adds the `port_forwarding` setting to the widget."""

        def on_switch_state(_, enabled: bool, toggle_widget: ToggleWidget):

            # When we start displaying port forwarding, we no longer want to be showing the
            # setup guide.
            display_port_forwarding = self._controller.feature_flags.get("DisplayPortForwarding")

            description_value = self.PORT_FORWARDING_DESCRIPTION
            if enabled:
                description_value = self.PORT_FORWARDING_SETUP_GUIDE
                if display_port_forwarding:
                    description_value = self.PORT_FORWARDING_DESCRIPTION_LEARN_MORE

            toggle_widget.save_setting(enabled)
            toggle_widget.description.set_label(description_value)

        self.port_forwarding = ToggleWidget(
            controller=self._controller,
            title=self.PORT_FORWARDING_LABEL,
            description=self.PORT_FORWARDING_DESCRIPTION,
            setting_name="settings.features.port_forwarding",
            requires_subscription_to_be_active=True,
            callback=on_switch_state
        )
        is_pf_enabled = self.port_forwarding.get_setting()
        display_port_forwarding = self._controller.feature_flags.get("DisplayPortForwarding")
        if is_pf_enabled:
            self.port_forwarding.description.set_label(
                self.PORT_FORWARDING_DESCRIPTION_LEARN_MORE
                if display_port_forwarding
                else self.PORT_FORWARDING_SETUP_GUIDE
            )

        self.append(self.port_forwarding)

    @GObject.Signal(name="netshield-setting-changed", arg_types=(int,))
    def netshield_setting_changed(self, custom_dns_enabled: int):
        """Signal emitted after a netshield setting is set."""

    def on_custom_dns_setting_changed(
        self, custom_dns_widget: CustomDNSWidget, custom_dns_enabled: int
    ):
        """temp"""
        def _on_dialog_button_click(confirmation_dialog: ConfirmationDialog, response_type: int):
            enable_custom_dns = Gtk.ResponseType(response_type) == Gtk.ResponseType.YES
            if enable_custom_dns:
                self.netshield.off()
                self._settings_window.notify_user_with_reconnect_message()
            else:
                # We need to reverse back the option here since gtk does not allow an easy way to
                # intercept changes before they happen.
                custom_dns_widget.off()

            confirmation_dialog.destroy()

        netshield_disabled = int(self.netshield.get_setting()) == NetShield.NO_BLOCK

        if not custom_dns_enabled or netshield_disabled:
            self._settings_window.notify_user_with_reconnect_message(
                only_notify_on_active_connection=True
            )
            return

        dialog = ConfirmationDialog(
            message=self._build_dialog_content(),
            title="Enable Custom DNS",
            yes_text="_Enable", no_text="_Cancel"
        )
        dialog.set_default_size(400, 200)
        dialog.connect("response", _on_dialog_button_click)
        dialog.set_modal(True)
        dialog.set_transient_for(self._settings_window)
        dialog.present()

    def _build_dialog_content(self):
        container = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        container.set_spacing(10)

        question = Gtk.Label(label="Enable Custom DNS ?")
        question.set_halign(Gtk.Align.START)

        clarification = Gtk.Label(label="This will disable Netshield.")
        clarification.set_halign(Gtk.Align.START)
        clarification.add_css_class("dim-label")

        learn_more = Gtk.Label(
            label='<a href="https://protonvpn.com/support/custom-dns#netshield">Learn more</a>'
        )
        learn_more.set_halign(Gtk.Align.START)
        learn_more.add_css_class("dim-label")
        learn_more.set_use_markup(True)

        container.append(question)
        container.append(clarification)
        container.append(learn_more)

        return container

    def build_split_tunneling(self):
        """Build split tunneling UI.
        """
        self.split_tunneling = SplitTunnelingToggle.build(self._controller)
        self.append(self.split_tunneling)
