﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/s3tables/S3Tables_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace S3Tables {
namespace Model {

/**
 * <p>Contains details about a schema field.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/s3tables-2018-05-10/SchemaField">AWS
 * API Reference</a></p>
 */
class SchemaField {
 public:
  AWS_S3TABLES_API SchemaField() = default;
  AWS_S3TABLES_API SchemaField(Aws::Utils::Json::JsonView jsonValue);
  AWS_S3TABLES_API SchemaField& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_S3TABLES_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>An optional unique identifier for the schema field. Field IDs are used by
   * Apache Iceberg to track schema evolution and maintain compatibility across
   * schema changes. If not specified, S3 Tables automatically assigns field IDs.</p>
   */
  inline int GetId() const { return m_id; }
  inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
  inline void SetId(int value) {
    m_idHasBeenSet = true;
    m_id = value;
  }
  inline SchemaField& WithId(int value) {
    SetId(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the field.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  SchemaField& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The field type. S3 Tables supports all Apache Iceberg primitive types. For
   * more information, see the <a
   * href="https://iceberg.apache.org/spec/#primitive-types">Apache Iceberg
   * documentation</a>.</p>
   */
  inline const Aws::String& GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  template <typename TypeT = Aws::String>
  void SetType(TypeT&& value) {
    m_typeHasBeenSet = true;
    m_type = std::forward<TypeT>(value);
  }
  template <typename TypeT = Aws::String>
  SchemaField& WithType(TypeT&& value) {
    SetType(std::forward<TypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A Boolean value that specifies whether values are required for each row in
   * this field. By default, this is <code>false</code> and null values are allowed
   * in the field. If this is <code>true</code> the field does not allow null
   * values.</p>
   */
  inline bool GetRequired() const { return m_required; }
  inline bool RequiredHasBeenSet() const { return m_requiredHasBeenSet; }
  inline void SetRequired(bool value) {
    m_requiredHasBeenSet = true;
    m_required = value;
  }
  inline SchemaField& WithRequired(bool value) {
    SetRequired(value);
    return *this;
  }
  ///@}
 private:
  int m_id{0};

  Aws::String m_name;

  Aws::String m_type;

  bool m_required{false};
  bool m_idHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_typeHasBeenSet = false;
  bool m_requiredHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3Tables
}  // namespace Aws
