﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/OperatorResponse.h>
#include <aws/ec2/model/ResponseLaunchTemplateData.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace EC2 {
namespace Model {

/**
 * <p>Describes a launch template version.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/LaunchTemplateVersion">AWS
 * API Reference</a></p>
 */
class LaunchTemplateVersion {
 public:
  AWS_EC2_API LaunchTemplateVersion() = default;
  AWS_EC2_API LaunchTemplateVersion(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_EC2_API LaunchTemplateVersion& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_EC2_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_EC2_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The ID of the launch template.</p>
   */
  inline const Aws::String& GetLaunchTemplateId() const { return m_launchTemplateId; }
  inline bool LaunchTemplateIdHasBeenSet() const { return m_launchTemplateIdHasBeenSet; }
  template <typename LaunchTemplateIdT = Aws::String>
  void SetLaunchTemplateId(LaunchTemplateIdT&& value) {
    m_launchTemplateIdHasBeenSet = true;
    m_launchTemplateId = std::forward<LaunchTemplateIdT>(value);
  }
  template <typename LaunchTemplateIdT = Aws::String>
  LaunchTemplateVersion& WithLaunchTemplateId(LaunchTemplateIdT&& value) {
    SetLaunchTemplateId(std::forward<LaunchTemplateIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the launch template.</p>
   */
  inline const Aws::String& GetLaunchTemplateName() const { return m_launchTemplateName; }
  inline bool LaunchTemplateNameHasBeenSet() const { return m_launchTemplateNameHasBeenSet; }
  template <typename LaunchTemplateNameT = Aws::String>
  void SetLaunchTemplateName(LaunchTemplateNameT&& value) {
    m_launchTemplateNameHasBeenSet = true;
    m_launchTemplateName = std::forward<LaunchTemplateNameT>(value);
  }
  template <typename LaunchTemplateNameT = Aws::String>
  LaunchTemplateVersion& WithLaunchTemplateName(LaunchTemplateNameT&& value) {
    SetLaunchTemplateName(std::forward<LaunchTemplateNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version number.</p>
   */
  inline long long GetVersionNumber() const { return m_versionNumber; }
  inline bool VersionNumberHasBeenSet() const { return m_versionNumberHasBeenSet; }
  inline void SetVersionNumber(long long value) {
    m_versionNumberHasBeenSet = true;
    m_versionNumber = value;
  }
  inline LaunchTemplateVersion& WithVersionNumber(long long value) {
    SetVersionNumber(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description for the version.</p>
   */
  inline const Aws::String& GetVersionDescription() const { return m_versionDescription; }
  inline bool VersionDescriptionHasBeenSet() const { return m_versionDescriptionHasBeenSet; }
  template <typename VersionDescriptionT = Aws::String>
  void SetVersionDescription(VersionDescriptionT&& value) {
    m_versionDescriptionHasBeenSet = true;
    m_versionDescription = std::forward<VersionDescriptionT>(value);
  }
  template <typename VersionDescriptionT = Aws::String>
  LaunchTemplateVersion& WithVersionDescription(VersionDescriptionT&& value) {
    SetVersionDescription(std::forward<VersionDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time the version was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreateTime() const { return m_createTime; }
  inline bool CreateTimeHasBeenSet() const { return m_createTimeHasBeenSet; }
  template <typename CreateTimeT = Aws::Utils::DateTime>
  void SetCreateTime(CreateTimeT&& value) {
    m_createTimeHasBeenSet = true;
    m_createTime = std::forward<CreateTimeT>(value);
  }
  template <typename CreateTimeT = Aws::Utils::DateTime>
  LaunchTemplateVersion& WithCreateTime(CreateTimeT&& value) {
    SetCreateTime(std::forward<CreateTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The principal that created the version.</p>
   */
  inline const Aws::String& GetCreatedBy() const { return m_createdBy; }
  inline bool CreatedByHasBeenSet() const { return m_createdByHasBeenSet; }
  template <typename CreatedByT = Aws::String>
  void SetCreatedBy(CreatedByT&& value) {
    m_createdByHasBeenSet = true;
    m_createdBy = std::forward<CreatedByT>(value);
  }
  template <typename CreatedByT = Aws::String>
  LaunchTemplateVersion& WithCreatedBy(CreatedByT&& value) {
    SetCreatedBy(std::forward<CreatedByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether the version is the default version.</p>
   */
  inline bool GetDefaultVersion() const { return m_defaultVersion; }
  inline bool DefaultVersionHasBeenSet() const { return m_defaultVersionHasBeenSet; }
  inline void SetDefaultVersion(bool value) {
    m_defaultVersionHasBeenSet = true;
    m_defaultVersion = value;
  }
  inline LaunchTemplateVersion& WithDefaultVersion(bool value) {
    SetDefaultVersion(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the launch template.</p>
   */
  inline const ResponseLaunchTemplateData& GetLaunchTemplateData() const { return m_launchTemplateData; }
  inline bool LaunchTemplateDataHasBeenSet() const { return m_launchTemplateDataHasBeenSet; }
  template <typename LaunchTemplateDataT = ResponseLaunchTemplateData>
  void SetLaunchTemplateData(LaunchTemplateDataT&& value) {
    m_launchTemplateDataHasBeenSet = true;
    m_launchTemplateData = std::forward<LaunchTemplateDataT>(value);
  }
  template <typename LaunchTemplateDataT = ResponseLaunchTemplateData>
  LaunchTemplateVersion& WithLaunchTemplateData(LaunchTemplateDataT&& value) {
    SetLaunchTemplateData(std::forward<LaunchTemplateDataT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The entity that manages the launch template.</p>
   */
  inline const OperatorResponse& GetOperator() const { return m_operator; }
  inline bool OperatorHasBeenSet() const { return m_operatorHasBeenSet; }
  template <typename OperatorT = OperatorResponse>
  void SetOperator(OperatorT&& value) {
    m_operatorHasBeenSet = true;
    m_operator = std::forward<OperatorT>(value);
  }
  template <typename OperatorT = OperatorResponse>
  LaunchTemplateVersion& WithOperator(OperatorT&& value) {
    SetOperator(std::forward<OperatorT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_launchTemplateId;

  Aws::String m_launchTemplateName;

  long long m_versionNumber{0};

  Aws::String m_versionDescription;

  Aws::Utils::DateTime m_createTime{};

  Aws::String m_createdBy;

  bool m_defaultVersion{false};

  ResponseLaunchTemplateData m_launchTemplateData;

  OperatorResponse m_operator;
  bool m_launchTemplateIdHasBeenSet = false;
  bool m_launchTemplateNameHasBeenSet = false;
  bool m_versionNumberHasBeenSet = false;
  bool m_versionDescriptionHasBeenSet = false;
  bool m_createTimeHasBeenSet = false;
  bool m_createdByHasBeenSet = false;
  bool m_defaultVersionHasBeenSet = false;
  bool m_launchTemplateDataHasBeenSet = false;
  bool m_operatorHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
