﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/apigateway/APIGatewayRequest.h>
#include <aws/apigateway/APIGateway_EXPORTS.h>
#include <aws/apigateway/model/PatchOperation.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace APIGateway {
namespace Model {

/**
 * <p>A request to change information about the BasePathMapping
 * resource.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/apigateway-2015-07-09/UpdateBasePathMappingRequest">AWS
 * API Reference</a></p>
 */
class UpdateBasePathMappingRequest : public APIGatewayRequest {
 public:
  AWS_APIGATEWAY_API UpdateBasePathMappingRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateBasePathMapping"; }

  AWS_APIGATEWAY_API Aws::String SerializePayload() const override;

  AWS_APIGATEWAY_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The domain name of the BasePathMapping resource to change.</p>
   */
  inline const Aws::String& GetDomainName() const { return m_domainName; }
  inline bool DomainNameHasBeenSet() const { return m_domainNameHasBeenSet; }
  template <typename DomainNameT = Aws::String>
  void SetDomainName(DomainNameT&& value) {
    m_domainNameHasBeenSet = true;
    m_domainName = std::forward<DomainNameT>(value);
  }
  template <typename DomainNameT = Aws::String>
  UpdateBasePathMappingRequest& WithDomainName(DomainNameT&& value) {
    SetDomainName(std::forward<DomainNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The identifier for the domain name resource. Supported only for private
   * custom domain names. </p>
   */
  inline const Aws::String& GetDomainNameId() const { return m_domainNameId; }
  inline bool DomainNameIdHasBeenSet() const { return m_domainNameIdHasBeenSet; }
  template <typename DomainNameIdT = Aws::String>
  void SetDomainNameId(DomainNameIdT&& value) {
    m_domainNameIdHasBeenSet = true;
    m_domainNameId = std::forward<DomainNameIdT>(value);
  }
  template <typename DomainNameIdT = Aws::String>
  UpdateBasePathMappingRequest& WithDomainNameId(DomainNameIdT&& value) {
    SetDomainNameId(std::forward<DomainNameIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The base path of the BasePathMapping resource to change.</p> <p>To specify an
   * empty base path, set this parameter to <code>'(none)'</code>.</p>
   */
  inline const Aws::String& GetBasePath() const { return m_basePath; }
  inline bool BasePathHasBeenSet() const { return m_basePathHasBeenSet; }
  template <typename BasePathT = Aws::String>
  void SetBasePath(BasePathT&& value) {
    m_basePathHasBeenSet = true;
    m_basePath = std::forward<BasePathT>(value);
  }
  template <typename BasePathT = Aws::String>
  UpdateBasePathMappingRequest& WithBasePath(BasePathT&& value) {
    SetBasePath(std::forward<BasePathT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For more information about supported patch operations, see <a
   * href="https://docs.aws.amazon.com/apigateway/latest/api/patch-operations.html">Patch
   * Operations</a>.</p>
   */
  inline const Aws::Vector<PatchOperation>& GetPatchOperations() const { return m_patchOperations; }
  inline bool PatchOperationsHasBeenSet() const { return m_patchOperationsHasBeenSet; }
  template <typename PatchOperationsT = Aws::Vector<PatchOperation>>
  void SetPatchOperations(PatchOperationsT&& value) {
    m_patchOperationsHasBeenSet = true;
    m_patchOperations = std::forward<PatchOperationsT>(value);
  }
  template <typename PatchOperationsT = Aws::Vector<PatchOperation>>
  UpdateBasePathMappingRequest& WithPatchOperations(PatchOperationsT&& value) {
    SetPatchOperations(std::forward<PatchOperationsT>(value));
    return *this;
  }
  template <typename PatchOperationsT = PatchOperation>
  UpdateBasePathMappingRequest& AddPatchOperations(PatchOperationsT&& value) {
    m_patchOperationsHasBeenSet = true;
    m_patchOperations.emplace_back(std::forward<PatchOperationsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_domainName;

  Aws::String m_domainNameId;

  Aws::String m_basePath;

  Aws::Vector<PatchOperation> m_patchOperations;
  bool m_domainNameHasBeenSet = false;
  bool m_domainNameIdHasBeenSet = false;
  bool m_basePathHasBeenSet = false;
  bool m_patchOperationsHasBeenSet = false;
};

}  // namespace Model
}  // namespace APIGateway
}  // namespace Aws
